import { assertNever, Promise } from '@mecontrol/web-inline';

interface ResolvedState {
    state: 'resolved';
    urgent: boolean;
}

interface IdlingState {
    state: 'idling';
    promise: Promise<boolean>;
    resolve: (urgent: boolean) => void;
    reject: (error: Error) => void;
    eventHandler: () => void;
}

interface UninitializedState {
    state: 'uninitialized';
}

type IdleState = UninitializedState | IdlingState | ResolvedState;

let idleState: IdleState = { state: 'uninitialized' };
export function onLoadUntilUrgent(
    document: Document = window.document
): Promise<boolean> {
    switch (idleState.state) {
        case 'resolved': {
            return Promise.resolve(idleState.urgent);
        }
        case 'idling': {
            return idleState.promise;
        }
        case 'uninitialized': {
            let resolve: IdlingState['resolve'], reject: IdlingState['reject'];
            const promise = new Promise<boolean>((resolver, rejecter) => {
                resolve = resolver;
                rejecter = rejecter;
            });

            if (document.readyState === 'complete') {
                resolve!(false);
                idleState = {
                    state: 'resolved',
                    urgent: false
                };
            } else {
                const eventHandler = () => {
                    idleState = { state: 'resolved', urgent: false };
                    resolve(idleState.urgent);
                    window.removeEventListener('load', eventHandler);
                };

                window.addEventListener('load', eventHandler);

                idleState = {
                    state: 'idling',
                    promise,
                    // Trust me TypeScript - this works
                    resolve: resolve!,
                    reject: reject!,
                    eventHandler
                };
            }

            return promise;
        }
        default: {
            let message = `Unexpected idle state: ${JSON.stringify(idleState)}`;
            assertNever(idleState, message);
        }
    }
}

export function setUrgent(): void {
    // if (idleState.state == 'idling') {
    //     window.removeEventListener('load', idleState.eventHandler);

    //     idleState.resolve(true);
    //     idleState = { state: 'resolved', urgent: true };
    // }

    switch (idleState.state) {
        case 'uninitialized':
            idleState = { state: 'resolved', urgent: true };
            break;
        case 'idling':
            window.removeEventListener('load', idleState.eventHandler);
            idleState.resolve(true);
            idleState = { state: 'resolved', urgent: true };
            break;
        case 'resolved':
            // Do nothing
            break;
        default:
            let message = `Unexpected idle state: ${JSON.stringify(idleState)}`;
            assertNever(idleState, message);
    }
}

/** TESTING ONLY */
export function resetIdleState(): void {
    idleState = { state: 'uninitialized' };
}
