// Regex that finds {#} so it can be replaced by the arguments in string format
const FORMAT_RE = /\{\d+\}/g;

// Regex that finds { and } so they can be removed on a lookup for string format
const FORMAT_ARGS_RE = /[\{\}]/g;

// Encode if a character not matches with [a-zA-Z0-9_{space}.,-].
const ENCODE_HTML_RE = /[^\w .,-]/g;

/**
 * Checks if the string starts with the provided prefix.
 * Usage: Strings.startsWith('abc', 'ab') returns true
 * @param str String to check
 * @param prefix Prefix to match
 * @returns True if the prefix is at the start of the string
 */
export function startsWith(str: string, prefix: string): boolean {
    return (str.substr(0, prefix.length) === prefix);
}

/**
 * Like C# string format.
 * Usage Example: "hello {0}!".format("mike") will return "hello mike!"
 * Calling format on a string with less arguments than specified in the format is invalid
 * Example "I love {0} every {1}".format("CXP") will result in a Debug Exception.
 * @param str The string format
 * @param params The string elements to be inserted on the given format
 */
export function format(str: string, ...params: string[]): string {
    // Callback match function
    function replace_func(match: string): string {
        // looks up in the params
        let replacement = params[parseInt(match.replace(FORMAT_ARGS_RE, ''), 10)];
        if (replacement == null) {
            replacement = '';
        }

        return replacement;
    }

    return (str.replace(FORMAT_RE, replace_func));
}

/**
 * Aggressively encodes a string to be displayed in the browser. All non-letter characters are converted
 * to their Unicode entity ref, e.g. &#65;, space, comma, and dash are left un-encoded as well.
 * Usage: _divElement.innerHTML =_someValue.encodeHtml());
 * @param str The string to be encoded
 * @returns HTML encoded string
 */
export function encodeHtml(str: string): string {
    if (!str) {
        return '';
    }

    let charCodeResult = {
        c: 0, // Code
        s: -1 // Next skip index
    };

    return str.replace(ENCODE_HTML_RE, (match: string, index: number, fullString: string): string => {
        if (extendedCharCodeAt(fullString, index, charCodeResult)) {
            return ['&#', charCodeResult.c, ';'].join('');
        }

        // If extendedCharCodeAt returns false that means this index is the low surrogate,
        // which has already been processed, so we remove it by returning an empty string.
        return '';
    });
}

/**
 * Gets the char code from str at index.
 * Supports Secondary-Multilingual-Plane Unicode characters (SMP), e.g. codes above 0x10000
 * @param str String to get the char code from
 * @param idx Index of the char to get code for
 * @param result Receives result code c: and index s: info
 */
function extendedCharCodeAt(str: string, idx: number, result: { c: number; s: number; }): boolean {
    let skip = (result.s === idx);
    if (!skip) {
        idx = idx || 0;
        let code = str.charCodeAt(idx);
        let hi, low;
        result.s = -1;
        if (code < 0xD800 || code > 0xDFFF) {
            // Main case, Basic-Multilingual-Plane (BMP) code points.
            result.c = code;
        }
        else if (code <= 0xDBFF) { // High surrogate of SMP
            hi = code;
            low = str.charCodeAt(idx + 1);
            result.c = ((hi - 0xD800) * 0x400) + (low - 0xDC00) + 0x10000;
            result.s = idx + 1;
        }
        else { // Low surrogate of SMP, 0xDC00 <= code && code <= 0xDFFF
            // Shouldn't really ever come in here, previous call to this method would set skip index in result
            // in high surrogate case, which is short-circuited at the start of this function.
            result.c = -1;
            skip = true;
        }
    }
    return !skip;
}
