import { AnyObject } from '@mecontrol/common';
import { hasOwn } from '@mecontrol/web-inline';

/**
 * Parses an object and cleans each field that matches the provided list. Also allows specifying certain
 * fields to never be scrubbed or scrubbing everything. If a match is itself an object, then the object's
 * fields are also cleaned. 'Cleaning' here means that the value is obfuscated by replacing it with a
 * string that states the type of value. Functions inside the object will always be scrubbed.
 * @param obj The object to remove fields from
 * @param fields List of field names. If a field matches the name, it will be scrubbed. If the name is
 * negated ('!fieldName'), it will override everything else and the field will not be scrubbed. Including
 * '*' in the list means scrubbing everything (save for negated fields). Not providing fields means the
 * same as including '*'.
 * @param filter Optional field that is hidden from outside callers that is used for recursive calls to
 * indicate if everything in a given object should be cleaned. For cases when a field to clean is actually
 * a full object.
 * @returns A copy of the provided object, with all fields scrubbed as determined by the provided fields
 */
export function filterObject(obj: AnyObject, fields: string[] = ['*'], filter?: boolean): object {
    let filteredObj: AnyObject = {};

    for (let propName in obj) {
        if (!hasOwn(obj, propName)) {
            continue;
        }

        let propValue = obj[propName];

        // Filter a property if:
        //  - The special flag is set
        //  - The special '*' field was provided
        //  - The property's name is in the list of fields
        // Override fitlering a property if:
        //  - The negated property's name is in the list of fields
        // If Array.indexOf is not availabel (ie8), then we default to filtering
        let filterProp = !fields.indexOf || (fields.indexOf('!' + propName) === -1 && (filter || fields.indexOf('*') !== -1 || fields.indexOf(propName) !== -1));
        let propType = typeof propValue;
        switch (propType) {
            case 'object':
                filteredObj[propName] = propValue !== null ? filterObject(propValue, fields, filterProp) : '*null*';
                break;
            case 'function':
                // Always filter functions
                filterProp = true;
            default:
                filteredObj[propName] = filterProp ? `*${propType}*` : propValue;
                break;
        }
    }

    return filteredObj;
}
