import { createError } from '@mecontrol/web-inline';

// Consider: Using [url-parse](https://npm.im/url-parse) if our url parsing needs grow more complex
// or we need a cross platform (i.e. non-browser) parser

/*
 * Our ParsedUrl does not implement every method of the [URL standard](https://developer.mozilla.org/en-US/docs/Web/API/URL).
 * The following properties are left out because they contain duplicate information
 * that is represented in other property, and we have not implemented the logic
 * to keep them up to date with each other (mostly for size reasons and because
 * we don't need them at this time).
 *
 * - host (duplicates hostname & port)
 * - href (duplicates every other property)
 * - origin (duplicates protocol, host, and port)
 * - search (duplicates searchParams)
 *
 * The following properties are not included because the <a> tag does not parse
 * and return them in IE. In other words, they are not supported in IE:
 *
 * - origin
 * - password
 * - username
 *
 */

/**
 * A mutable object representing the parts of a URL, inspired by the [URL standard](https://developer.mozilla.org/en-US/docs/Web/API/URL).
 */
export interface ParsedUrl {
    /** The protocol scheme of the URL, including the final ':'. */
    protocol: string;

    /** The domain of the URL */
    hostname: string;

    /** The port of the URL */
    port: string;

    /** An initial '/' followed by the path of the URL. */
    pathname: string;

    /** An object containing the GET query arguments contained in the URL. */
    searchParams: SearchParams;

    /** The URL fragment, including the leading '#' */
    hash: string;

    /**
     * Serialize the given url to a string. This method will not add the port
     * number for http and https protocols if the port is the well-known port
     * for those protocols.
     * @param url The ParsedUrl to serialize
     */
    toString(): string;
}

// Inspired by [URLSearchParams standard](https://developer.mozilla.org/en-US/docs/Web/API/URLSearchParams)
/** The SearchParams interface defines utility methods to work with the query string of a URL. */
export interface SearchParams {
    /**
     * Appends a specified key/value pair as a new search parameter.
     * @param name The key of the search parameter to append
     * @param value The value of the search parameter to append
     */
    append(name: string, value: string): void;

    /**
     * Deletes the given search parameter, and its associated value, from the list of all search parameters.
     * @param name The key to delete
     */
    delete(name: string): void;

    /**
     * Returns the first value associated to the given search parameter.
     * @param name The key of the search param to get
     * @returns The string value of the search param or null if the key does not exist
     */
    get(name: string): string | null;

    /**
     * Returns all the values association with a given search parameter.
     * @param name The key to retreive
     * @returns The array of all values associated with the given key, or an empty array if the key does not exist
     */
    getAll(name: string): string[];

    /**
     * Returns a Boolean indicating if such a search parameter exists.
     * @param name The key to check
     */
    has(name: string): boolean;

    /**
     * Sets the value associated to a given search parameter to the given value. If there were several values, delete the others.
     * @param name The key to set
     * @param value The value to set the search parameter to
     */
    set(name: string, value: string): void;

    /**
     * Returns a string containing a query string suitable for use in a URL.
     */
    toString(): string;
}

/**
 * Parse a URL into its parts. This method also parses the queryString into an object
 * @param url The URL to parse
 */
export function parseUrl(url: string): ParsedUrl {
    if (!url) {
        throw createError(`Invalid string given to parseUrl: "${url}"`);
    }

    return new Url(url);
}

export function isJavaScriptScheme(parsedUrl: ParsedUrl): boolean {
    return parsedUrl.protocol == "javascript:";
}

class Url implements ParsedUrl {

    private originalUrl: string;

    public protocol: string;

    public hostname: string;

    public port: string;

    public pathname: string;

    public searchParams: SearchParams;

    public hash: string;

    constructor(url: string) {
        this.originalUrl = url;

        const a = document.createElement('a');
        a.href = url;

        // IE workaround - IE doesn't fill in the page's protocol, hostname or port properties for relative URLs.
        // However it does construct an absolute URL when accessing the .href property for relative URLs. So this
        // line reads the href property of the A tag which will always be absolute in all browsers tested and
        // re-assigns it to the A tag so that the absolute URL is parsed.
        a.href = a.href;

        this.protocol = a.protocol;
        this.hostname = a.hostname;
        this.port = a.port;
        this.pathname = a.pathname;
        this.searchParams = new UrlSearchParams(a.search);
        this.hash = a.hash;
    }

    public toString(): string {
        if (this.protocol == 'javascript:') {
            return this.originalUrl;
        }

        // IE workaround - IE always specifies the port for known protocols
        // even if it is not specified in the original URL
        let port = "";
        if (
            this.port &&
            !(this.port === '443' && this.protocol === 'https:') &&
            !(this.port === '80' && this.protocol === 'http:')
        ) {
            port = ":" + this.port;
        }

        // IE workaround - IE doesn't begin `pathname` with a '/' for absolute http(s) URLs
        const path = !isJavaScriptScheme(this) && this.pathname[0] !== '/' ? '/' + this.pathname : this.pathname;
        const search = this.searchParams ? this.searchParams.toString() : "";
        const protocolSeparator = !isJavaScriptScheme(this) ? "//" : "";

        return `${this.protocol}${protocolSeparator}${this.hostname}${port}${path}${search}${this.hash}`;
    }
}

/**
 * Replace '+' characters with a space and decode a string using decodeURIComponent
 * @param s The string to decode
 */
function decode(s: string): string {
    return decodeURIComponent(s.replace(/\+/g, ' '));
}

const queryStringParser = /([^=?&]+)=?([^&]*)/g;

class UrlSearchParams implements SearchParams {

    // IE9 - Object.create is only available in IE9+
    // Using Object.create(null):
    // - https://stackoverflow.com/a/32263086
    // - https://davidwalsh.name/object-create-null
    private queryParams: Record<string, string[]> = Object.create(null);

    constructor(queryString: string) {
        let match;
        // tslint:disable-next-line:no-conditional-assignment
        while (match = queryStringParser.exec(queryString)) {
            // match[0] is the entire matched string. match[1] is first group, match[2] is second, ...
            const key = decode(match[1]);
            const value = decode(match[2]);

            this.append(key, value);
        }
    }

    public append(key: string, value: string): void {
        if (key in this.queryParams) {
            this.queryParams[key].push(value);
        }
        else {
            this.queryParams[key] = [value];
        }
    }

    public delete(key: string): void {
        delete this.queryParams[key];
    }

    public get(key: string): string | null {
        return key in this.queryParams ? this.queryParams[key][0] : null;
    }

    public getAll(key: string): string[] {
        return key in this.queryParams ? [...this.queryParams[key]] : [];
    }

    public has(key: string): boolean {
        return key in this.queryParams;
    }

    public set(key: string, value: string): void {
        this.queryParams[key] = [value];
    }

    public toString(): string {
        const serializedPairs = [];

        // tslint:disable-next-line:forin // We are creating queryParams as an object with no prototype so hasOwn check is unnecessary
        for (let key in this.queryParams) {
            const values = this.queryParams[key];
            for (let i = 0; i < values.length; i++) {
                serializedPairs.push(encodeURIComponent(key) + '=' + encodeURIComponent(values[i]));
            }
        }

        return serializedPairs.length ? '?' + serializedPairs.join('&') : '';
    }
}
