import { StringMap } from "@mecontrol/common/";
import { hasOwn } from '@mecontrol/web-inline';
import { parseUrl, isJavaScriptScheme } from './url.polyfill';

/**
 * Sets a dictionary of query string parameters to a given url. It overrides any existing
 * query parameters on the URL with the same name
 * @param url The url to set the parameters to.
 * @param queryParams The query parameters to set on the url.
 * @param includeEmptys Default true; determines whether to include query param if string value is empty or null
 * @returns The url with the given queryParameters set
 */
export function setQueryParams(url: string, queryParams: Partial<StringMap>, includeEmptys: boolean = true): string {
    const parsedUrl = parseUrl(url);

    // Technically, the URL spec allows javascript schemes to have query params
    // but in actuality, query params can break javascript schemes,
    // so we don't set query params for javascript schemes
    if (!isJavaScriptScheme(parsedUrl)) {
        for (let key in queryParams) {
            let qvalue = queryParams[key];
            if (hasOwn(queryParams, key) && (includeEmptys || !!qvalue)) {
                parsedUrl.searchParams.set(key, qvalue || '');
            }
        }
    }

    return parsedUrl.toString();
}

/**
 * Checks to see if the given URL has the given queryParam.
 *
 * @param url {string} The url to check
 * @param queryParam {string} The query param key name to check for in the given url
 */
export function hasQueryParam(url: string, queryParam: string): boolean {
    if (!url) {
        return false;
    }

    const parsedUrl = parseUrl(url);

    // Technically, the URL spec allows javascript schemes to have query params
    // but in actuality, query params can break javascript schemes,
    // so we don't set query params for javascript schemes
    if (isJavaScriptScheme(parsedUrl)) {
        return false;
    }

    return parsedUrl.searchParams.has(queryParam);
}

/**
 * Get the hostname of a URL
 * @param url The URL to get the hostname of
 */
export function getHostname(url: string): string {
    return parseUrl(url).hostname;
}


export function urlEncodeArr(inputArr: Uint8Array) {
    return base64EncArr(inputArr)
        .replace(/=/g, "")
        .replace(/\+/g, "-")
        .replace(/\//g, "_");
}

function base64EncArr(aBytes: Uint8Array) {
    const eqLen = (3 - (aBytes.length % 3)) % 3;
    let sB64Enc = "";

    for (var nMod3, nLen = aBytes.length, nUint24 = 0, nIdx = 0; nIdx < nLen; nIdx++) {
        nMod3 = nIdx % 3;
        /* Uncomment the following line in order to split the output in lines 76-character long: */
        /*
        if (nIdx > 0 && (nIdx * 4 / 3) % 76 === 0) { sB64Enc += "\r\n"; }
        */
        nUint24 |= aBytes[nIdx] << (16 >>> nMod3 & 24);
        if (nMod3 === 2 || aBytes.length - nIdx === 1) {
            sB64Enc += String.fromCharCode(
                uint6ToB64(nUint24 >>> 18 & 63),
                uint6ToB64(nUint24 >>> 12 & 63),
                uint6ToB64(nUint24 >>> 6 & 63),
                uint6ToB64(nUint24 & 63)
            );
            nUint24 = 0;
        }
    }
    if (eqLen === 0) { return sB64Enc; }
    var sb64Val = sB64Enc.substring(0, sB64Enc.length - eqLen)
    if (eqLen === 1) {
        return sb64Val + "="
    }
    else {
        return sb64Val + "=="
    }
}

function uint6ToB64 (nUint6: number) {
    if (nUint6 < 26) { return nUint6 + 65; }
    else if (nUint6 < 52) { return nUint6 + 71; }
    else if (nUint6 < 62) { return nUint6 - 4; }
    else if (nUint6 === 62) { return 43; }
    else if (nUint6 === 63) { return 47; }
    else { return 65; }
}