import { IPartnerApiCallEvent, IPartnerApiCallParams, IPartnerApiCallPartC, AnyObject } from '@mecontrol/common';
import { ME, EventMoment, createPartA, scrubUrl } from '@mecontrol/web-inline';
import { filterObject } from '@mecontrol/web-boot';

const eventName = 'Ms.Webi.MeControl.PartnerApiCall';
// Filter practically everything
const FILTER_FIELDS = ['*', '!idp', '!authenticatedState', '!type', '!apiGeneration'];
const URL_PATTERN = /https?:\/\/.*/i;

/**
 * Creates a PartnerApiCall event to log information on public Me Control API usage from partners.
 * Filters out user data before logging the information.
 * @param apiParams The object containig details of the API call a partner made
 * @returns The PartnerApiCall event that was created
 */
export function createPartnerApiCall(apiParams: IPartnerApiCallParams, moment: EventMoment): IPartnerApiCallEvent {
    let partA = createPartA(eventName, moment.date, moment.cv);

    let filteredParams = filterObject(apiParams.params, FILTER_FIELDS);
    filterUrlsInObject(filteredParams);

    let partC: IPartnerApiCallPartC = {
        baseType: undefined,
        partner: ME.Config.ptn,
        controlVersion: ME.Config.ver,
        methodName: apiParams.method,
        parameters: JSON.stringify(filteredParams)
    };

    return {
        partA,
        partB: undefined,
        partC
    };
}

/**
 * Modifies the provided object by combing through all its properties in search of strings that
 * match the URL pattern regex. These strings are then scrubbed of all their query parameters.
 * Also walks through any nested objects the original object may have.
 * @param targetObj The object to scrub
 */
function filterUrlsInObject(targetObj: AnyObject): void {
    // tslint:disable-next-line:forin
    for (let propName in targetObj) {
        let propValue = targetObj[propName];
        switch (typeof propValue) {
            case 'object':
                if (propValue != null) {
                    filterUrlsInObject(propValue);
                }
                break;
            case 'string':
                if (URL_PATTERN.test(propValue)) {
                    targetObj[propName] = scrubUrl(propValue);
                }
                break;
        }
    }
}
